using System;
using System.Text.RegularExpressions;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Diagnostics;
using System.Web.RegularExpressions;


namespace CSharpRecipes
{
	public class RegEx
    {
        #region "10.1 Przetwarzanie cigw speniajcych warunki wyraenia regularnego"

        public static void TestFindSubstrings()
        {
            string matchPattern = "<.*>";

            string source = @"<?xml version='1.0' encoding='UTF-8'?>
    <!-- Mj komentarz -->
             <![CDATA[<escaped> <><chars>>>>>]]>
             <Window ID='Main'>
               <Control ID='TextBox'>
                 <Property Top='0' Left='0' Text='BLANK'/>
               </Control>
               <Control ID='Label'>
               <Property Top='0' Left='0' Caption='Tu wpisz nazwisko'/>
               </Control>
               <Control ID='Label'>
               <Property Top='0' Left='0' Caption='Tu wpisz nazwisko'/>
               </Control>
             </Window>";

            Console.WriteLine("NIEPOWTARZALNE PODCIGI");
            Match[] x1 = FindSubstrings(source, matchPattern, true);
            foreach (Match m in x1)
            {
                Console.WriteLine(m.Value);
            }

            Console.WriteLine();
            Console.WriteLine("WSZYSTKIE PODCIGI");
            Match[] x2 = FindSubstrings(source, matchPattern, false);
            foreach (Match m in x2)
            {
                Console.WriteLine(m.Value);
            }
        }


public static Match[] FindSubstrings(string source, string matchPattern, 
                                      bool findAllUnique)
{
    SortedList uniqueMatches = new SortedList();
    Match[] retArray = null;
    Regex RE = new Regex(matchPattern, RegexOptions.Multiline);
    MatchCollection theMatches = RE.Matches(source);
    if (findAllUnique)
    {
        for (int counter = 0; counter < theMatches.Count; counter++)
        {
            if (!uniqueMatches.ContainsKey(theMatches[counter].Value))
            {
                uniqueMatches.Add (theMatches [counter].Value, theMatches[counter]);
            }
        }
        retArray = new Match[uniqueMatches.Count];
        uniqueMatches.Values.CopyTo(retArray, 0);
    }
    else
    {
        retArray = new Match[theMatches.Count];
        theMatches.CopyTo(retArray, 0);
    }
    return (retArray);
}
		
		#endregion

#region "10.2 Wyodrbnianie grup z obiektu MatchCollection"
public static void TestExtractGroupings()
		{
            string source = @"Path = ""\\MjSerwer\MojaUsuga\Mojacieka;
                               \\MjSerwer2\MojaUsuga2\Mojacieka2\""";
            string matchPattern = @"\\\\(?<Serwer>\w*)\\(?<Usuga>\w*)\\";

//			foreach(Match m in theMatches)
//			{
//				for (int counter = 0; counter < m.Groups.Count; counter++)
//				{
//					Console.WriteLine(m.Groups[0].GroupNameFromNumber(counter), m.Groups[counter]);
//				}
//			}


			foreach (Dictionary<string, Group> grouping in ExtractGroupings(source, matchPattern, false))
			{
				foreach (KeyValuePair<string, Group> kvp in grouping)
                    Console.WriteLine("Indeks / Warto = " + kvp.Key + " / " + kvp.Value);
                Console.WriteLine("");
			}
		}

		public static List<Dictionary<string, Group>> ExtractGroupings(string source, string matchPattern, 
													   bool wantInitialMatch)
		{
			List<Dictionary<string, Group>> keyedMatches = new List<Dictionary<string, Group>>();
			int startingElement = 1;
			if (wantInitialMatch)
			{
				startingElement = 0;
			}

			Regex RE = new Regex(matchPattern, RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace);
			MatchCollection theMatches = RE.Matches(source);

//			return (theMatches);
			
			
			foreach(Match m in theMatches)
			{
				Dictionary<string, Group> groupings = new Dictionary<string, Group>();

				for (int counter = startingElement; counter < m.Groups.Count; counter++)
				{
					groupings.Add(RE.GroupNameFromNumber(counter), m.Groups[counter]);
				}

				keyedMatches.Add(groupings);
			}

			return (keyedMatches);
		}
        #endregion

        #region "10.3 Weryfikacja skadni wyraenia regularnego"

        public static void TestUserInputRegEx(string regEx)
		{
			if (VerifyRegEx(regEx))
                Console.WriteLine("To wyraenie regularne jest prawidowe.");
			else
                Console.WriteLine("To wyraenie regularne nie jest prawidowe.");
		}

		public static bool VerifyRegEx(string testPattern)
		{
			bool isValid = true;

			if ((testPattern != null) && (testPattern.Trim().Length > 0))
			{
				try
				{
					Regex.Match("", testPattern);
				}
				catch (ArgumentException ae)
				{
                    // NIEWACIWY WZORZEC: bd syntaktyczny
					isValid = false;
					Console.WriteLine(ae);
				}
			}
			else
			{
                // NIEWACIWY WZORZEC: brak wzorca lub wzorzec pusty
				isValid = false;
			}

			return (isValid);
		}
 
        #endregion

        #region "10.4 Szybki sposb wyszukiwania ostatniego podcigu speniajcego kryteria"
        public static void TestFindLast()
		{
			Match theMatch = Regex.Match("jeden dwa trzy dwa jeden", "dwa", RegexOptions.RightToLeft);
			
			Regex RE = new Regex("dwa", RegexOptions.RightToLeft);
			theMatch = RE.Match("jeden dwa trzy dwa jeden");
			
			Console.WriteLine(theMatch.Value);
		}
		#endregion

        #region "10.5 Zastpowanie znakw lub sw w cigu znakw"
        // Staa definiujca zestaw opcji dla wyraenia regularnego
		const RegexOptions defaultOptions = RegexOptions.IgnorePatternWhitespace | 
			RegexOptions.Multiline;

		public static string Replace(string source, char matchPattern, string replaceStr)
		{
			return (Replace(source, matchPattern.ToString(), replaceStr, -1, 0, 
				defaultOptions));
		}

		public static string Replace(string source, char matchPattern, string replaceStr, 
			int count)
		{
			return (Replace(source.ToString(), matchPattern.ToString(), replaceStr, 
				count, 0, defaultOptions));
		}

		public static string Replace(string source, char matchPattern, string replaceStr, 
			int count, int startPos)
		{
			return (Replace(source.ToString(), matchPattern.ToString(), replaceStr, 
				count, startPos, defaultOptions));
		}

		public static string Replace(string source, char matchPattern, string replaceStr, 
			int count, int startPos, RegexOptions options)
		{
			return (Replace(source.ToString(), matchPattern.ToString(), replaceStr, 
				count, startPos, options));
		}

		public static string Replace(string source, string matchPattern, string replaceStr)
		{
			return (Replace(source, matchPattern, replaceStr, -1, 0, defaultOptions));
		}
    
		public static string Replace(string source, string matchPattern, string replaceStr, 
			int count)
		{
			return (Replace(source, matchPattern, replaceStr, count, 0, 
				defaultOptions));
		}

		public static string Replace(string source, string matchPattern, string replaceStr, 
			int count, int startPos)
		{
			return (Replace(source, matchPattern, replaceStr, count, startPos, 
				defaultOptions));
		}

		public static string Replace(string source, string matchPattern, string replaceStr, 
			int count, int startPos, RegexOptions options)
		{
			Regex RE = new Regex(matchPattern, options);
			string newString = RE.Replace(source, replaceStr, count, startPos);
  
			return (newString);
		}

		public static void TestReplace()
		{
            string source = "Zastp tekst FOO w tym bloku tekstw FOO."; 
            string matchPattern = "FOO";
			string replaceStr = "BAR";

			Console.WriteLine(Replace(source, matchPattern, replaceStr));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, -1));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, -1, 0, defaultOptions));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, 1));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, 1, 0, defaultOptions));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, 1));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, 1, 20, defaultOptions));

			Console.WriteLine(Replace(source, matchPattern, replaceStr, -1, 0, defaultOptions));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, 1, 0, defaultOptions));
			Console.WriteLine(Replace(source, matchPattern, replaceStr, 1, 20, defaultOptions));
		}
        #endregion

        #region "10.6 Ulepszanie prostej funkcji do zastpowania cigw znakw"
        public static string MatchHandler(Match theMatch)
		{
			// Obsuga waciwoci Top znacznika Property
			if (theMatch.Value.StartsWith("<Property"))
			{
				long topPropertyValue = 0;

                // Uzyskanie liczbowej wartoci atrybutu Top
				Match topPropertyMatch = Regex.Match(theMatch.Value, "Top=\"([-]*\\d*)");
				if (topPropertyMatch.Success)
				{
					if (topPropertyMatch.Groups[1].Value.Trim().Equals(""))
					{
                        // Jeli jest pusta, ustawienie na warto zero.
						return (theMatch.Value.Replace("Top=\"\"", "Top=\"0\""));
					}
					else if (topPropertyMatch.Groups[1].Value.Trim().Equals("-"))
					{
                        // W przypadku znaku minus (bd skadni), ustawienie na warto zero.
						return (theMatch.Value.Replace("Top=\"-\"", "Top=\"0\""));
					}
					else
					{
                        // Prawidowa liczba.
                        // Konwersja cigu na warto liczbow.

                        topPropertyValue = long.Parse(topPropertyMatch.Groups[1].Value, 
							System.Globalization.NumberStyles.Any);

                        // Jeli atrybut Top znajduje si poza okrelinym zakresem, 
                        // ustawienie go na zero.

						if (topPropertyValue < 0 || topPropertyValue > 5000)
						{
							return (theMatch.Value.Replace("Top=\"" + topPropertyValue + "\"", 
								"Top=\"0\""));
						}
					}
				}
			}

			return (theMatch.Value);
		}

		public static void ComplexReplace(string matchPattern, string source)
		{
			MatchEvaluator replaceCallback = new MatchEvaluator(MatchHandler);
			string newString = Regex.Replace(source, matchPattern, replaceCallback);

            Console.WriteLine("Zastpiony cig znakw = " + newString); 
		}

		public static void TestComplexReplace()
		{
			string matchPattern = "<.*>";
			string source = @"<?xml version=""1.0\"" encoding=\""UTF-8\""?>
        <Window ID=""Main"">
            <Control ID=""TextBox"">
                <Property Top=""100"" Left=""0"" Text=""BLANK""/>
            </Control>
            <Control ID=""Label"">
                <Property Top=""99990"" Left=""0"" Caption=""Tu wpisz nazwisko""/>
            </Control>
        </Window>";

			ComplexReplace(matchPattern, source);
		}
        #endregion

        #region "10.7 A Implementacja lepszego tokenizera"
        public static string[] Tokenize(string equation)
		{
			Regex RE = new Regex(@"([\+\-\*\(\)\^\\])");
			return (RE.Split(equation));
		}

		public static void TestTokenize()
		{
			foreach(string token in Tokenize("(y - 3)(3111*x^21 + x + 320)"))
                Console.WriteLine("Znacznik = " + token.Trim());
		}
        #endregion

        #region "10.8 Kompilacja wyrae regularnych"
        public static void TestRegExCompilation()
		{
			RegexCompilationInfo[] RE = new RegexCompilationInfo[2] {new RegexCompilationInfo("PATTERN", RegexOptions.Compiled, "CompiledPATTERN", 
						"Chapter_Code", true), new RegexCompilationInfo("NAME", RegexOptions.Compiled, "CompiledNAME", "Chapter_Code", true)};

			System.Reflection.AssemblyName AName = new System.Reflection.AssemblyName();
			AName.Name = "REGEX_Test";

			Regex.CompileToAssembly(RE, AName);
		}
		
		public static void CreateRegExDLL(string assmName)
		{
			RegexCompilationInfo[] RE = new RegexCompilationInfo[2] {
				new RegexCompilationInfo("PATTERN", RegexOptions.Compiled, "CompiledPATTERN", "Chapter_Code", true), 
				new RegexCompilationInfo("NAME", RegexOptions.Compiled, "CompiledNAME", "Chapter_Code", true)};

			System.Reflection.AssemblyName aName = new System.Reflection.AssemblyName();
			aName.Name = assmName;
			
			Regex.CompileToAssembly(RE, aName);
			
			System.Threading.Thread.Sleep(2000);
		}
		
		public static void UseRegExDLL()
		{
			//Chapter_Code.CompiledNAME CN = new Chapter_Code.CompiledNAME();
            //Match MName = CN.Match("Pobierz wyraenie NAME z tekstu.");
			//Console.WriteLine("MName.Value = " + MName.Value);
		}
        #endregion

        #region "10.9 Zliczanie wierszy tekstu"
        public static long LineCount(string source, bool isFileName)
		{
			// Uruchomienie kodu licznika czasu...
			Counter c = new Counter();
			c.Clear();
			c.Start();
            // Uruchomienie kodu licznika czasu...
			
			
			if (source != null)
			{
				string text = source;

				if (isFileName)
				{
					FileStream FS = new FileStream(source, FileMode.Open, 
						FileAccess.Read, FileShare.Read);
					StreamReader SR = new StreamReader(FS);
					text = SR.ReadToEnd();
					SR.Close();
					FS.Close();
				}

				Regex RE = new Regex("\n", RegexOptions.Multiline);
				MatchCollection theMatches = RE.Matches(text);

				if (isFileName)
					Console.WriteLine("LineCount: " + (theMatches.Count).ToString());
				else
					Console.WriteLine("LineCount: " + (theMatches.Count + 1).ToString());
						
			
				// Koniec kodu licznika czasu...
				c.Stop();
				Console.WriteLine("Sekund: " + c.Seconds.ToString());
				// Koniec kodu licznika czasu...


				// Obsuga plikw o zerowej dugoci
				//   Zwrmy uwag, e cig znakw zawsze koczy si znakiem koca wiersza i dlatego
				//        zawsze bdzie mia dugo 1 lub wicej
				if (isFileName)
				{
					return (theMatches.Count);
				}
				else
				{
					return (theMatches.Count) + 1;
				}
			}
			else
			{
				// Koniec pomiaru czasu...
				c.Stop();
				Console.WriteLine("Sekund: " + c.Seconds.ToString());
				// Koniec kodu licznika czasu...


                //  Obsuga przypadku pustego argumentu source.
				return (0);
			}
		}

		public static long LineCount2(string source, bool isFileName)
		{
			// Pocztek pomiaru czasu...
			Counter c = new Counter();
			c.Clear();
			c.Start();
			// Pocztek pomiaru czasu...


			if (source != null)
			{
				string text = source;
				long numOfLines = 0;
				
				if (isFileName)
				{
					FileStream FS = new FileStream(source, FileMode.Open, 
						FileAccess.Read, FileShare.Read);
					StreamReader SR = new StreamReader(FS);
					
					while (text != null)
					{
						text = SR.ReadLine();
						
						if (text != null)
						{
							++numOfLines;
						}
					}
					
					SR.Close();
					FS.Close();

					Console.WriteLine("LineCount: " + numOfLines.ToString());
			
			
					// Koniec pomiaru czasu...
					c.Stop();
					Console.WriteLine("Sekund: " + c.Seconds.ToString());
					// Koniec pomiaru czasu...

					
					return (numOfLines);
				}
				else
				{
					Regex RE = new Regex("\n", RegexOptions.Multiline);
					MatchCollection theMatches = RE.Matches(text);

					Console.WriteLine("LineCount: " + (theMatches.Count + 1).ToString());
			
			
					// Koniec pomiaru czasu...
					c.Stop();
					Console.WriteLine("Sekund: " + c.Seconds.ToString());
					// Koniec pomiaru czasu...


					return (theMatches.Count + 1);
				}
			}
			else
			{
				// Koniec pomiaru czasu...
				c.Stop();
				Console.WriteLine("Sekund: " + c.Seconds.ToString());
				// Koniec pomiaru czasu...
			
			
				// Obsuga przypadku pustrego parametru source
				return (0);
			}
		}

		public static void TestLineCount()
		{
            //  Obliczenie liczby wierszy w pliku tekstowym TestFile.txt.
			LineCount(@"..\..\TestFile.txt", true);
			Console.WriteLine();

            //  Obliczenie liczby wierszy w cigu znakw TestString
            LineCount("Wiersz1\r\nWiersz2\r\nWiersz3\nWiersz4", false);
			Console.WriteLine();

            // Obliczenie liczby wierszy w cigu znakw TestString
			LineCount("", false);
			Console.WriteLine();

            // Obliczenie liczby wierszy w pliku tekstowym TestFile.txt
			LineCount2(@"..\..\TestFile.txt", true);
			Console.WriteLine();

            // Obliczenie liczby wierszy w cigu znakw TestString
            LineCount2("Wiersz1\r\nWiersz2\r\nWiersz3\nWiersz4", false);
			Console.WriteLine();

            // Obliczenie liczby wierszy w cigu znakw TestString
			LineCount2("", false);
		}
        #endregion

        #region "10.10 Zwracanie caych wierszy w przypadku znalezienia podcigu pasujcego do wzorca"
        public static List<string> GetLines2(string source, string pattern, bool isFileName)
		{
			// Pocztek pomiaru czasu...
			Counter c = new Counter();
			c.Clear();
			c.Start();
			// Pocztek pomiaru czasu...
			
			
			string text = source;
			List<string> matchedLines = new List<string>();

            // Pobranie caego tekstu pliku, jeli argument source oznacza plik.
			if (isFileName)
			{
				FileStream FS = new FileStream(source, FileMode.Open, 
					FileAccess.Read, FileShare.Read);
				StreamReader SR = new StreamReader(FS);
					
				while (text != null)
				{
					text = SR.ReadLine();

					if (text != null)
					{
                        // Uruchomienie wyraenia regularnego dla wszystkich wierszy w cigu.
						Regex RE = new Regex(pattern, RegexOptions.Multiline);
						MatchCollection theMatches = RE.Matches(text);

						if (theMatches.Count > 0)
						{
                            // Pobranie wiersza w przypadku znalezienia pasujcego podcigu.
							matchedLines.Add(text);
						}
					}
				}
					
				SR.Close();
				FS.Close();
			}
			else
			{
                // Uruchomienie wyraenia regularnego raz dla caego cigu znakw.
				Regex RE = new Regex(pattern, RegexOptions.Multiline);
				MatchCollection theMatches = RE.Matches(text);

                // Pobranie wiersza dla kadego pasujcego podcigu.
				foreach (Match m in theMatches)
				{
					int lineStartPos = GetBeginningOfLine(text, m.Index);
					int lineEndPos = GetEndOfLine(text, (m.Index + m.Length - 1));
					string line = text.Substring(lineStartPos, lineEndPos - lineStartPos);
					matchedLines.Add(line);
				}
			}
			
	
			// Koniec pomiaru czasu...
			c.Stop();
			Console.WriteLine("Sekund: " + c.Seconds.ToString());
			// Koniec pomiaru czasu...


			return (matchedLines);
		}


		public static List<string> GetLines(string source, string pattern, bool isFileName)
		{
			// Pocztek pomieru czasu...
			Counter c = new Counter();
			c.Clear();
			c.Start();
			// Pocztek pomiaru czasu...
			
			
			string text = source;
			List<string> matchedLines = new List<string>();
			
			// Jeli to jest plik, pobranie jego caej zawartoci.
			if (isFileName)
			{
				FileStream FS = new FileStream(source, FileMode.Open, 
					FileAccess.Read, FileShare.Read);
				StreamReader SR = new StreamReader(FS);
				text = SR.ReadToEnd();
				SR.Close();
				FS.Close();
			}

            //  Uruchomienie wyraenia regularnego raz dla caego cigu znakw.
			Regex RE = new Regex(pattern, RegexOptions.Multiline);
			MatchCollection theMatches = RE.Matches(text);

            // Pobranie wiersza dla kadego pasujcego podcigu.
			foreach (Match m in theMatches)
			{
				int lineStartPos = GetBeginningOfLine(text, m.Index);
				int lineEndPos = GetEndOfLine(text, (m.Index + m.Length - 1));
				string line = text.Substring(lineStartPos, lineEndPos - lineStartPos);
				matchedLines.Add(line);
			}

	
			// Koniec pomiaru czasu...
			c.Stop();
			Console.WriteLine("Sekund: " + c.Seconds.ToString());
			// Koniec pomiaru czasu...


			return (matchedLines);
		}

		public static int GetBeginningOfLine(string text, int startPointOfMatch)
		{
			if (startPointOfMatch > 0)
			{
				--startPointOfMatch;
			}
			
			if (startPointOfMatch >= 0 && startPointOfMatch < text.Length)
			{
                //  Przesunicie w lewo do momentu znalezienia pierwszego znaku '\n.
				for (int index = startPointOfMatch; index >= 0; index--)
				{
					if (text[index] == '\n')
					{
						return (index + 1);
					}
				}
				
				return (0);
			}
			
			return (startPointOfMatch);
		}
		
		public static int GetEndOfLine(string text, int endPointOfMatch)
		{
			if (endPointOfMatch >= 0 && endPointOfMatch < text.Length)
			{
                //  Przesunicie w prawo do momentu znalezienia pierwszego znaku '\n.
				for (int index = endPointOfMatch; index < text.Length; index++)
				{
					if (text[index] == '\n')
					{
						return (index);
					}
				}
				
				return (text.Length);
			}
			
			return (endPointOfMatch);
		}
		
		public static void TestGetLine()
		{
			// Pobranie pasujcych wierszy w obrbie pliku TestFile.txt
			Console.WriteLine("\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
			List<string> lines = GetLines(@"..\..\TestFile.txt", "\n", true);
			foreach (string s in lines)
				Console.WriteLine("MatchedLine: " + s);

            // Pobranie pasujcych wierszy w obrbie cigu znakw TestString
			Console.WriteLine("\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
            lines = GetLines("Wiersz1\r\nWiersz2\r\nWiersz3\nWiersz4", "Wiersz", false);
			foreach (string s in lines)
				Console.WriteLine("MatchedLine: " + s);

            // Pobranie pasujcych wierszy w obrbie cigu znakw TestString
			Console.WriteLine("\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
			lines = GetLines("\rLLLLLL", "L", false);
			foreach (string s in lines)
				Console.WriteLine("MatchedLine: " + s);

            // Pobranie pasujcych wierszy w obrbie pliku TestFile.txt
			Console.WriteLine("\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
			lines = GetLines2(@"..\..\TestFile.txt", "\n", true);
			foreach (string s in lines)
				Console.WriteLine("MatchedLine: " + s);

            // Pobranie pasujcych wierszy w obrbie cigu znakw TestString
			Console.WriteLine("\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
            lines = GetLines2("Wiersz1\r\nWiersz2\r\nWiersz3\nWiersz4", "Wiersz", false);
			foreach (string s in lines)
				Console.WriteLine("MatchedLine: " + s);

            // Pobranie pasujcych wierszy w obrbie cigu znakw TestString
			Console.WriteLine("\n\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
			lines = GetLines2("\rLLLLLL", "L", false);
			foreach (string s in lines)
				Console.WriteLine("MatchedLine: " + s);
		}
		#endregion

        #region "10.11 Wyszukiwanie okrelonego wystpienia pasujcego podcigu"
        public static Match FindOccurrenceOf(string source, string pattern, int occurrence)
		{
			if (occurrence < 1)
			{
                throw (new ArgumentException("Argument nie moe by mniejszy ni 1", "occurrence"));
			}

            //  Zaczynamy od indeksu 0.
			--occurrence;

            // Uruchomienie wyraenia regularnego raz dla caego cigu znakw.
			Regex RE = new Regex(pattern, RegexOptions.Multiline);
			MatchCollection theMatches = RE.Matches(source);

			if (occurrence >= theMatches.Count)
			{
				return (null);
			}
			else
			{
				return (theMatches[occurrence]);
			}
		}

		public static List<Match> FindEachOccurrenceOf(string source, string pattern, int occurrence)
		{
			List<Match> occurrences = new List<Match>();

            // Uruchomienie wyraenia regularnego raz dla cigu source.
			Regex RE = new Regex(pattern, RegexOptions.Multiline);
			MatchCollection theMatches = RE.Matches(source);

			for (int index = (occurrence - 1); index < theMatches.Count; index += occurrence)
			{
				occurrences.Add(theMatches[index]);
			}

			return (occurrences);
		}
		
		public static void TestOccurrencesOf()
		{
            Match matchResult = FindOccurrenceOf("jeden dwa trzy jeden dwa trzy jeden dwa trzy jeden"
                            + " dwa trzy jeden dwa trzy jeden dwa trzy", "dwa", 2);

			if (matchResult != null)
				Console.WriteLine(matchResult.ToString() + "\t" + matchResult.Index);

			Console.WriteLine();
            List<Match> results = FindEachOccurrenceOf("jeden jeden dwa trzy jeden dwa trzy jeden"
                            + " dwa trzy jeden dwa trzy", "jeden", 2);

			foreach (Match m in results)
				Console.WriteLine(m.ToString() + "\t" + m.Index);
		}
		#endregion

        #region "10.12 Wykorzystanie czsto uywanych wzorcw"
        /*
			Wzorzec pasujcy do cigw zoonych ze znakw alfanumerycznych, 
            symboli -, +, . oraz spacji:
			^([\w\.+-]|\s)*$

			Wyraenie pasujce do znakw alfanumerycznych zawierajcych 
            znaki -, +,  oraz spacje, przy zaoeniu, e w cigu wystpuje co 
            najmniej jeden taki znak i nie wicej ni 10:
            ^([\w\.+-]|\s){1,10}$


			Wyraenie pasujce do daty w formacie ##/##/####, gdzie dzie i 
            miesic moe by cyfr skadajc si z jednej lub dwch cyfr, 
            natomiast rok moe skada si z dwch bd z czterech cyfr:
            ^\d{1,2}\/\d{1,2}\/\d{2,4}$
        
			Wzorzec godziny z opcjonalnym przyrostkiem am bd pm 
            (zwrmy uwag, e wzorzec ten pasuje rwnie do czasu w formacie 
            wojskowym):
            ^\d{1,2}:\d{2}\s?([ap]m)?$

			Wzorzec adresu IP w standardzie IPv4:
            ^([0-2]?[0-5]?[0-5]\.){3}[0-2]?[0-5]?[0-5]$


			Wzorzec adresu email w formacie nazwisko@adres, 
            gdzie adres nie jest adresem IP:
            ^[A-Za-z0-9_\-\.]+@(([A-Za-z0-9\-])+\.)+([A-Za-z\-])+$


			Wzorzec adresu email w formacie nazwisko@adres, gdzie 
            adres jest adresem IP:
            ^[A-Za-z0-9_\-\.]+@([0-2]?[0-5]?[0-5]\.){3}[0-2]?[0-5]?[0-5]$


			Wzorzec kwoty pieninej z opcjonalnym symbolem dolara ($) oraz 
            prefiksem w postaci znakw + lub - (zwrmy uwag, e mona 
            wprowadzi dowoln liczb cyfr po przecinku):
            ^\$?[+-]?[\d,]*(\.\d*)?$


			Wyraenie regularne podobne do poprzedniego, z t rnic, e 
            zezwala na wystpienie nie wicej ni dwch cyfr po przecinku:
            ^\$?[+-]?[\d,]*\.?\d{0,2}$

            Wyraenie pasujce do numeru karty kredytowej w postaci czterech 
            grup po cztery cyfry oddzielonych spacj, znakiem - lub bez znaku 
            separatora:
            ^((\d{4}[- ]?){3}\d{4})$


			Amerykaski kod pocztowy w postaci piciu cyfr z opcjonalnym 
            czterocyfrowym rozszerzeniem:
            ^\d{5}(-\d{4})?$


			Numer telefonu w formacie pnocnoamerykaskim z opcjonalnym numerem 
            kierunkowym i opcjonalnym znakiem -, bez numeru wewntrznego:
            ^(\(?[0-9]{3}\)?)?\-?[0-9]{3}\-?[0-9]{4}$

			Wyraenie pasujce do numeru telefonu podobne do poprzedniego wyraenia 
            regularnego, ale z opcjonalnym piciocyfrowym numerem wewntrznym 
            poprzedzonym sowem ext lub extension:
            ^(\(?[0-9]{3}\)?)?\-?[0-9]{3}\-?[0-9]{4}(\s*ext(ension)?[0-9]{5})?$


			Pena cieka do pliku zawierajca liter napdu i opcjonalnie nazw 
            pliku wraz z trzyznakowym rozszerzeniem (zwrmy uwag, e wzorzec 
            nie zezwala na wystpienie znakw .. oznaczajcych przejcie o jeden 
            katalog w gr hierarchii plikw, ani znaku . w nazwie katalogu 
            oddzielajcego nazw zasadnicz od rozszerzenia).
            ^[a-zA-Z]:[\\/]([_a-zA-Z0-9]+[\\/]?)*([_a-zA-Z0-9]+\.[_a-zA-Z0-9]{0,3})?$

		*/
        #endregion

        #region "10.13 Dokumentowanie wyrae regularnych"
        public static void RegExComments()
		{
            //Wprowadzenie komentarzy w wyraeniach regularnych z wykorzystaniem znaku #:
            string matchPattern = @"\\\\                 # Wyszukaj:  \\
                        (?<TheServer>\w*)    # Nazwa serwera
                        \\                   # Wyszukaj:  \
                        (?<TheService>\w*)\\ # Nazwa usugi";

            // W przypadku uycia tego wyraenia w obiekcie Regex, do parametru options 
            // konstruktora obiektu Regex naley doda element typu wyliczeniowego RegexOptions.IgnorePatternWhitespace:
            Regex RE = new Regex(matchPattern, 
                RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace);
            // wykorzystanie w standardowy sposb
            MatchCollection theMatches = RE.Matches("Tutaj powinien si znale tekst rdowy...");
        }
        #endregion

        #region "10.14 Zastosowanie wbudowanych wyrae regularnych do analizy stron ASP.NET"
        public static void TestASPNETParsing()
		{
			string testHTML = "<%-- Comment --%>  <%@ Page Language=\"CS\" " + 
							  "AutoEventWireup=\"false\" CodeFile=\"Default.aspx.cs\" " + 
							  "Inherits=\"Default_aspx\" %>" + 
							  "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" " + 
							  "\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">" +
							  "<html xmlns=\"http://www.w3.org/1999/xhtml\">  " + 
							  "<!-- #include file=\"MyFile.inc\" -->" + 
							  "<head runat=\"server\"> " +
							  "<%-- Comment --%>  <%$ expression  prefix:Value %>  <%= WebResource(\"resourc_name\") %>" + 
							  "<title>Untitled Page</title>  " + 
							  "<% Code %>  <%= expression %>  <%# Data Binding %>  " + 
							  "</head><body><form id=\"form1\" runat=\"server\"><div> " + 
							  "<asp:Login ID=\"Login1\" runat=\"server\"></asp:Login></div>" + 
							  "</form></body></html>";

			ASPNETStartEndTagParsing(testHTML);
		}
		
		public static void ASPNETStartEndTagParsing(string html)
		{
			int index = 0;
			while (index < html.Length)
			{
				Match m = null;

				AspExprRegex aspExpr = new AspExprRegex();
				m = aspExpr.Match(html, index);
				if (m.Success)
				{
					index = m.Index + m.Length;
					Console.WriteLine("Wyraenie ASP.NET");
					Console.WriteLine(m.Value);
					continue;
				}

				CommentRegex aspComment = new CommentRegex();
				m = aspComment.Match(html, index);
				if (m.Success)
				{
					index = m.Index + m.Length;
					Console.WriteLine("Komentarz ASP.NET");
					Console.WriteLine(m.Value);
					continue;
				}

				// Wywietlenie znacznika kocowego
				EndTagRegex aspEndTag = new EndTagRegex();
				m = aspEndTag.Match(html, index);
				if (m.Success)
				{
					index = m.Index + m.Length;
					Console.WriteLine("Kocowy znacznik ASP.NET");
					Console.WriteLine(m.Value);
					continue;
				}

				// Wywietlenie znacznika pocztkowego
				TagRegex aspTag = new TagRegex();
				m = aspTag.Match(html, index);
				if (m.Success)
				{
					index = m.Index + m.Length;
					Console.WriteLine("Znacznik pocztkowy ASP.NET");
					Console.WriteLine(m.Value);

					RunatServerRegex aspRunAt = new RunatServerRegex();
					Match mInner = aspRunAt.Match(m.Value, 0);
					if (mInner.Success)
					{
						//index = mInner.Index + mInner.Length;
						Console.WriteLine("\tASP.NET RunAt");
						Console.WriteLine("\t" + mInner.Value);
					}
					
					continue;
				}
				
				// Jeli doszlimy do tego miejsca, tzn. nie znaleziono pasujcych cigw, inkrementacja indeksu
				index++;
			}
			
			Console.WriteLine("Zakoczono");
		}
		
		
		public static void ExamineBuiltInASPNETParsingRegEx()
		{
			X0 testX0 = new X0();
			testX0.foo();

			X1 testX1 = new X1();
			testX1.foo();

			X4 testX4 = new X4();
			testX4.foo();

			X5 testX5 = new X5();
			testX5.foo();

			X6 testX6 = new X6();
			testX6.foo();

			X7 testX7 = new X7();
			testX7.foo();

			X8 testX8 = new X8();
			testX8.foo();

			X12 testX12 = new X12();
			testX12.foo();

			X13 testX13 = new X13();
			testX13.foo();

			X14 testX14 = new X14();
			testX14.foo();

			X16 testX16 = new X16();
			testX16.foo();

			X17 testX17 = new X17();
			testX17.foo();

			X18 testX18 = new X18();
			testX18.foo();

			X19 testX19 = new X19();
			testX19.foo();

			X20 testX20 = new X20();
			testX20.foo();
		}
		
		
		public class X0 : System.Web.RegularExpressions.AspCodeRegex
		{
			public void foo()
			{
				Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: AspCodeRegex");
			}
		}

		public class X1 : System.Web.RegularExpressions.AspExprRegex
		{
			public void foo()
			{
				Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: AspExprRegex");
			}
		}

		public class X4 : System.Web.RegularExpressions.CommentRegex
		{
			public void foo()
			{
				Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: CommentRegex");
			}
		}

		public class X5 : System.Web.RegularExpressions.DatabindExprRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: DatabindExprRegex");
			}
		}

		public class X6 : System.Web.RegularExpressions.DataBindRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: DataBindRegex");
			}
		}

		public class X7 : System.Web.RegularExpressions.DirectiveRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: DirectiveRegex");
			}
		}

		public class X8 : System.Web.RegularExpressions.EndTagRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: EndTagRegex");
			}
		}

		public class X12 : System.Web.RegularExpressions.GTRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: GTRegex");
			}
		}

		public class X13 : System.Web.RegularExpressions.IncludeRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: IncludeRegex");
			}
		}

		public class X14 : System.Web.RegularExpressions.LTRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: LTRegex");
			}
		}

		public class X16 : System.Web.RegularExpressions.RunatServerRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: RunatServerRegex");
			}
		}

		public class X17 : System.Web.RegularExpressions.ServerTagsRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: ServerTagsRegex");
			}
		}

		public class X18 : System.Web.RegularExpressions.SimpleDirectiveRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: SimpleDirectiveRegex");
			}
		}

		public class X19 : System.Web.RegularExpressions.TagRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: TagRegex");
			}
		}

		public class X20 : System.Web.RegularExpressions.TextRegex
		{
			public void foo()
			{
                Console.WriteLine("WZORZEC: " + this.pattern + "\tFOR: TextRegex");
			}
		}
		#endregion
	}
	
	
	
	#region Klasa pomocnicza obsugi pomiaru czasu
	// Kod pobrany z internetu...
	public class Counter 
	{
		long elapsedCount = 0;
		long startCount = 0;

		public void Start()
		{
			startCount = 0;
			QueryPerformanceCounter(ref startCount);
		}
		
		public void Stop()
		{
			long stopCount = 0;
			QueryPerformanceCounter(ref stopCount);

			elapsedCount += (stopCount - startCount);
		}

		public void Clear()
		{
			elapsedCount = 0;
		}

		public float Seconds
		{
			get
			{
				long freq = 0;
				QueryPerformanceFrequency(ref freq);
				return((float) elapsedCount / (float) freq);
			}
		}

		public override string ToString()
		{
			return String.Format("{0} sekund", Seconds);
		}

		static long Frequency 
		{
			get 
			{
				long freq = 0;
				QueryPerformanceFrequency(ref freq);
				return freq;
			}
		}
		static long Value 
		{
			get 
			{
				long count = 0;
				QueryPerformanceCounter(ref count);
				return count;
			}
		}

		[System.Runtime.InteropServices.DllImport("KERNEL32")]
		private static extern bool QueryPerformanceCounter(  ref long lpPerformanceCount);

		[System.Runtime.InteropServices.DllImport("KERNEL32")]
		private static extern bool QueryPerformanceFrequency( ref long lpFrequency);                     
	}
	#endregion
}
